function [DM_WCE, reject_fix_b, reject_wce_N, DM_WPE,...
    reject_fix_m, reject_wpe_N] = dm(d, wceband, wpeband)

% Performs the one-sided DM test at 5% level

% INPUT
% d:          Loss differential
% wceband:    Bandwidth for the WCE (default is floor(T^(1/2)))
% wpeband:    Bandwidth for the WPE (default is floor(T^(1/3)))

% OUTPUT
% DM_WCE:       DM test statistic using WCE of LR variance
% reject_fix_b: counter of rejections using WCE and fixed-b asymptotics
% reject_wce_N: counter of rejections using WCE and standard asymptotics
% DM_WPE:       DM test statistic using WPE of LR variance
% reject_fix_m: counter of rejections using WPE and fixed-m asymptotics
% reject_wpe_N: counter of rejections using WPE and standard asymptotics

T = size(d,1);

if exist('wceband', 'var') == 0
    wceband = floor(T^(1/2));
end

if exist('wpeband', 'var') == 0
    wpeband = floor(T^(1/3));
end

%% DM using fixed-b asymptotics
WCE = NeweyWest(d,wceband,1);
DM_WCE = sqrt(T)* mean(d) / sqrt(WCE);

% quantiles WCE
b = wceband/T;
CV_fix_b = 1.6449 + 2.1859*b + 0.3142*b^2 - 0.3427*b^3;
reject_fix_b  =  DM_WCE > CV_fix_b;
reject_wce_N =  DM_WCE> icdf('normal',0.95,0,1);


%% DM using fixed-m asymptotics

%discrete fourier transform of d
AA = dftmtx(size(d, 1));
w= AA*d;
pdm = w.* conj(w)/ (2*pi*T);

WPE = sum(pdm(2:wpeband+1,:))/wpeband * 2*pi;
DM_WPE = sqrt(T)* mean(d)/ sqrt(WPE);

reject_fix_m  =  DM_WPE > icdf('t',0.95,2*wpeband);
reject_wpe_N  =  DM_WPE> icdf('normal',0.95,0,1);

end